<?php
// logs.php
date_default_timezone_set('Asia/Phnom_Penh');
// Start session at the very beginning
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include authentication functions and database
// require_once 'includes/auth.php';
require_once 'config/database.php';
require_once 'config/session.php';
require_once 'log_functions.php';


// Check if user has permission to view logs
if (!isLoggedIn()) {
    header('Location: index.php');
    exit();
}

checkRole(['cashier', 'admin', 'owner']);

if($_SESSION['user_role'] !== 'admin'){
    // return redirectByRole();
     header('Location: index.php');
    exit();
}

// Handle filters
$filters = [];
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    if (!empty($_GET['user_id'])) $filters['user_id'] = (int)$_GET['user_id'];
    if (!empty($_GET['action_type'])) $filters['action_type'] = $_GET['action_type'];
    if (!empty($_GET['module'])) $filters['module'] = $_GET['module'];
    if (!empty($_GET['status'])) $filters['status'] = $_GET['status'];
    if (!empty($_GET['date_from'])) $filters['date_from'] = $_GET['date_from'];
    if (!empty($_GET['date_to'])) $filters['date_to'] = $_GET['date_to'];
    if (!empty($_GET['search'])) $filters['search'] = $_GET['search'];
}

// Get logs
$page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
$limit = 50;
$offset = ($page - 1) * $limit;

$logs = $logger->getLogs($filters, $limit, $offset);

// Get total logs count WITH FILTERS
function getTotalLogsCountWithFilters($pdo, $filters = [])
{
    $where = [];
    $params = [];

    if (!empty($filters['user_id'])) {
        $where[] = "user_id = ?";
        $params[] = $filters['user_id'];
    }

    if (!empty($filters['action_type'])) {
        $where[] = "action_type = ?";
        $params[] = $filters['action_type'];
    }

    if (!empty($filters['module'])) {
        $where[] = "module = ?";
        $params[] = $filters['module'];
    }

    if (!empty($filters['status'])) {
        $where[] = "status = ?";
        $params[] = $filters['status'];
    }

    if (!empty($filters['date_from'])) {
        $where[] = "DATE(created_at) >= ?";
        $params[] = $filters['date_from'];
    }

    if (!empty($filters['date_to'])) {
        $where[] = "DATE(created_at) <= ?";
        $params[] = $filters['date_to'];
    }

    if (!empty($filters['search'])) {
        $where[] = "(description LIKE ? OR request_url LIKE ?)";
        $search_term = "%{$filters['search']}%";
        $params[] = $search_term;
        $params[] = $search_term;
    }

    $where_clause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

    $sql = "SELECT COUNT(*) as total FROM system_logs $where_clause";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['total'] ?? 0;
}

$total_logs = getTotalLogsCountWithFilters($pdo, $filters);
$total_pages = ceil($total_logs / $limit);

// Get users for filter dropdown
$users_stmt = $pdo->prepare("SELECT id, username, full_name FROM users ORDER BY username");
$users_stmt->execute();
$users = $users_stmt->fetchAll();

// Get unique action types and modules
$actions_stmt = $pdo->prepare("SELECT DISTINCT action_type FROM system_logs ORDER BY action_type");
$actions_stmt->execute();
$action_types = $actions_stmt->fetchAll();

$modules_stmt = $pdo->prepare("SELECT DISTINCT module FROM system_logs ORDER BY module");
$modules_stmt->execute();
$modules = $modules_stmt->fetchAll();

// Get statistics for dashboard
$today_logs = $pdo->prepare("SELECT COUNT(*) as count FROM system_logs WHERE DATE(created_at) = CURDATE()");
$today_logs->execute();
$today_logs_count = $today_logs->fetch()['count'];

$today_errors = $pdo->prepare("SELECT COUNT(*) as count FROM system_logs WHERE DATE(created_at) = CURDATE() AND status = 'error'");
$today_errors->execute();
$today_errors_count = $today_errors->fetch()['count'];
?>

<!DOCTYPE html>
<html lang="en">

<head>
    <?php
    require_once 'config/database.php';

    $system_stmt = $pdo->prepare("SELECT * FROM shop_info");
    $system_stmt->execute();
    $system = $system_stmt->fetch(PDO::FETCH_ASSOC); // Changed from fetchAll() to fetch()
    ?>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link rel="shortcut icon" href="assets/images/<?= $system['logo'] ?? 'logo.png' ?>" type="image/x-icon">
    <link rel="stylesheet" href="assets/css/all.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <link rel="stylesheet" href="assets/fontawesome/font-awesome.min.css">

    <link rel="stylesheet" href="assets/bootstrap/bootstrap.min.css">
    <link rel="stylesheet" href="assets/datatable/datatables.min.css">

    <title>System Logs - <?= $system['shop_name'] ?? 'Souvenir Shop' ?></title>
    <style>
        .log-success {
            color: #198754;
        }

        .log-error {
            color: #dc3545;
        }

        .log-warning {
            color: #ffc107;
        }

        .log-info {
            color: #0dcaf0;
        }

        .log-data {
            max-height: 200px;
            overflow-y: auto;
            background: #f8f9fa;
            padding: 10px;
            border-radius: 5px;
            font-family: monospace;
            font-size: 12px;
        }

        .action-badge {
            font-size: 0.8em;
            padding: 2px 8px;
        }

        .stats-card {
            border-radius: 10px;
            margin-bottom: 20px;
            box-shadow: 0 2px 5px rgba(0, 0, 0, 0.1);
        }

        #logs_table tr th {
            color: white;
        }

        /* .modal-backdrop .fade{
            display: none !important;
        } */
    </style>
</head>

<body>
    <!-- <? #php include 'includes/header.php'; 
            ?> -->

    <div class="container-fluid mt-4">
        <div class="row mb-3">
            <div class="col-12">
                <a id="back_home" class="btn btn-sm btn-primary"><i class="fa fa-home" aria-hidden="true"></i> Back home</a>
            </div>
        </div>
        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card stats-card bg-primary text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h6 class="card-title">Total Logs</h6>
                                <h2><?= number_format($total_logs) ?></h2>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-history fa-3x opacity-50"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card stats-card bg-info text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h6 class="card-title">Today's Logs</h6>
                                <h2><?= number_format($today_logs_count) ?></h2>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-calendar-day fa-3x opacity-50"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card stats-card bg-danger text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h6 class="card-title">Today's Errors</h6>
                                <h2><?= number_format($today_errors_count) ?></h2>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-exclamation-triangle fa-3x opacity-50"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card stats-card bg-success text-white">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h6 class="card-title">Total Pages</h6>
                                <h2><?= number_format($total_pages) ?></h2>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-file-alt fa-3x opacity-50"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <div class="d-flex justify-content-between align-items-center">
                            <h4 class="card-title m-0">
                                <i class="fas fa-history"></i> System Logs
                            </h4>
                            <div>
                                <!-- <button class="btn btn-sm btn-warning" onclick="exportLogs()">
                                    <i class="fas fa-download"></i> Export
                                </button> -->
                                <?php if (!empty($filters)): ?>
                                    <span class="badge bg-info ms-2">Filtered Results</span>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <div class="card-body">
                        <!-- Filters -->
                        <form method="GET" class="mb-4">
                            <div class="row g-3">
                                <div class="col-md-3">
                                    <label class="form-label">User</label>
                                    <select name="user_id" class="form-select">
                                        <option value="">All Users</option>
                                        <?php foreach ($users as $user): ?>
                                            <option value="<?= $user['id'] ?>" <?= isset($filters['user_id']) && $filters['user_id'] == $user['id'] ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($user['username']) ?> (<?= htmlspecialchars($user['full_name']) ?>)
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Action Type</label>
                                    <select name="action_type" class="form-select">
                                        <option value="">All Actions</option>
                                        <?php foreach ($action_types as $type): ?>
                                            <option value="<?= htmlspecialchars($type['action_type']) ?>" <?= isset($filters['action_type']) && $filters['action_type'] == $type['action_type'] ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($type['action_type']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Module</label>
                                    <select name="module" class="form-select">
                                        <option value="">All Modules</option>
                                        <?php foreach ($modules as $module): ?>
                                            <option value="<?= htmlspecialchars($module['module']) ?>" <?= isset($filters['module']) && $filters['module'] == $module['module'] ? 'selected' : '' ?>>
                                                <?= htmlspecialchars($module['module']) ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">Status</label>
                                    <select name="status" class="form-select">
                                        <option value="">All Status</option>
                                        <option value="success" <?= isset($filters['status']) && $filters['status'] == 'success' ? 'selected' : '' ?>>Success</option>
                                        <option value="error" <?= isset($filters['status']) && $filters['status'] == 'error' ? 'selected' : '' ?>>Error</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <label class="form-label">Date Range</label>
                                    <div class="input-group">
                                        <input type="date" name="date_from" class="form-control" value="<?= htmlspecialchars($filters['date_from'] ?? '') ?>">
                                        <span class="input-group-text">to</span>
                                        <input type="date" name="date_to" class="form-control" value="<?= htmlspecialchars($filters['date_to'] ?? '') ?>">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <label class="form-label">Search</label>
                                    <input type="text" name="search" class="form-control" placeholder="Search in description or URL..." value="<?= htmlspecialchars($filters['search'] ?? '') ?>">
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">&nbsp;</label>
                                    <button type="submit" class="btn btn-primary w-100">
                                        <i class="fas fa-filter"></i> Filter
                                    </button>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">&nbsp;</label>
                                    <a href="logs.php" class="btn btn-secondary w-100">
                                        <i class="fas fa-redo"></i> Reset
                                    </a>
                                </div>
                                <div class="col-md-2">
                                    <label class="form-label">&nbsp;</label>
                                    <button type="button" class="btn btn-outline-info w-100" onclick="clearFilters()">
                                        <i class="fas fa-times"></i> Clear
                                    </button>
                                </div>
                            </div>
                        </form>

                        <!-- Results Summary -->
                        <?php if (!empty($filters) || $total_logs > 0): ?>
                            <div class="alert alert-info mb-3">
                                <i class="fas fa-info-circle"></i>
                                Showing <?= count($logs) ?> of <?= number_format($total_logs) ?> logs
                                <?php if (!empty($filters)): ?>
                                    (filtered)
                                <?php endif; ?>
                            </div>
                        <?php endif; ?>

                        <!-- Logs Table -->
                        <div class="table-responsive">
                            <table class="table table-hover table-striped" id="logs_table">
                                <thead class="table-dark">
                                    <tr>
                                        <th>ID</th>
                                        <th>Date & Time</th>
                                        <th>User</th>
                                        <th>Action</th>
                                        <th>Module</th>
                                        <th>Description</th>
                                        <th>Status</th>
                                        <th>IP Address</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php if (empty($logs)): ?>
                                        <tr>
                                            <td colspan="9" class="text-center py-4">
                                                <div class="alert alert-warning m-0">
                                                    <i class="fas fa-exclamation-circle"></i> No logs found
                                                    <?php if (!empty($filters)): ?>
                                                        with the current filters
                                                    <?php endif; ?>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php else: ?>
                                        <?php foreach ($logs as $key => $log): ?>
                                            <tr>
                                                <td><strong><?= $key + 1 ?></strong></td>
                                                <td>
                                                    <small><?= date('d M Y', strtotime($log['created_at'])) ?></small><br>
                                                    <small class="text-muted"><?= date('h:i:s A', strtotime($log['created_at'])) ?></small>
                                                </td>
                                                <td>
                                                    <?php if ($log['username']): ?>
                                                        <div class="fw-bold"><?= htmlspecialchars($log['username']) ?></div>
                                                        <small class="text-muted"><?= htmlspecialchars($log['full_name'] ?? '') ?></small>
                                                    <?php else: ?>
                                                        <span class="text-muted fst-italic">System</span>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <span class="badge bg-info action-badge"><?= htmlspecialchars($log['action_type']) ?></span>
                                                </td>
                                                <td>
                                                    <span class="badge bg-secondary action-badge"><?= htmlspecialchars($log['module']) ?></span>
                                                </td>
                                                <td>
                                                    <div class="text-truncate" style="max-width: 300px;" title="<?= htmlspecialchars($log['description']) ?>">
                                                        <?= htmlspecialchars($log['description']) ?>
                                                    </div>
                                                </td>
                                                <td>
                                                    <?php if ($log['status'] === 'success'): ?>
                                                        <span class="badge bg-success rounded-pill">
                                                            <i class="fas fa-check"></i> Success
                                                        </span>
                                                    <?php else: ?>
                                                        <span class="badge bg-danger rounded-pill">
                                                            <i class="fas fa-times"></i> Error
                                                        </span>
                                                        <?php if ($log['error_message']): ?>
                                                            <br>
                                                            <button class="btn btn-sm btn-link p-0 text-danger"
                                                                onclick="showErrorMessage('<?= addslashes(htmlspecialchars($log['error_message'])) ?>')"
                                                                title="View error details">
                                                                <small>View Error</small>
                                                            </button>
                                                        <?php endif; ?>
                                                    <?php endif; ?>
                                                </td>
                                                <td>
                                                    <small class="text-monospace"><?= htmlspecialchars($log['ip_address']) ?></small>
                                                </td>
                                                <td>
                                                    <button class="btn btn-sm btn-info" onclick="showLogDetails(<?= htmlspecialchars(json_encode($log)) ?>)">
                                                        <i class="fas fa-eye"></i> Details
                                                    </button>
                                                </td>
                                            </tr>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Pagination -->
                        <?php if ($total_pages > 1): ?>
                            <nav aria-label="Logs pagination">
                                <ul class="pagination justify-content-end">
                                    <!-- Previous button -->
                                    <li class="page-item <?= $page <= 1 ? 'disabled' : '' ?>">
                                        <a class="page-link" href="?page=<?= $page - 1 ?><?= !empty($filters) ? '&' . http_build_query($filters) : '' ?>">
                                            <i class="fas fa-chevron-left"></i> Previous
                                        </a>
                                    </li>

                                    <!-- First page -->
                                    <?php if ($page > 3): ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=1<?= !empty($filters) ? '&' . http_build_query($filters) : '' ?>">1</a>
                                        </li>
                                        <?php if ($page > 4): ?>
                                            <li class="page-item disabled">
                                                <span class="page-link">...</span>
                                            </li>
                                        <?php endif; ?>
                                    <?php endif; ?>

                                    <!-- Page numbers around current -->
                                    <?php for ($i = max(1, $page - 2); $i <= min($total_pages, $page + 2); $i++): ?>
                                        <li class="page-item <?= $page == $i ? 'active' : '' ?>">
                                            <a class="page-link" href="?page=<?= $i ?><?= !empty($filters) ? '&' . http_build_query($filters) : '' ?>">
                                                <?= $i ?>
                                            </a>
                                        </li>
                                    <?php endfor; ?>

                                    <!-- Last page -->
                                    <?php if ($page < $total_pages - 2): ?>
                                        <?php if ($page < $total_pages - 3): ?>
                                            <li class="page-item disabled">
                                                <span class="page-link">...</span>
                                            </li>
                                        <?php endif; ?>
                                        <li class="page-item">
                                            <a class="page-link" href="?page=<?= $total_pages ?><?= !empty($filters) ? '&' . http_build_query($filters) : '' ?>">
                                                <?= $total_pages ?>
                                            </a>
                                        </li>
                                    <?php endif; ?>

                                    <!-- Next button -->
                                    <li class="page-item <?= $page >= $total_pages ? 'disabled' : '' ?>">
                                        <a class="page-link" href="?page=<?= $page + 1 ?><?= !empty($filters) ? '&' . http_build_query($filters) : '' ?>">
                                            Next <i class="fas fa-chevron-right"></i>
                                        </a>
                                    </li>
                                </ul>
                            </nav>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Log Details Modal -->
    <div class="modal fade" id="logDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Log Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div id="logDetailsContent"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Error Message Modal -->
    <div class="modal fade" id="errorMessageModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header bg-danger text-white">
                    <h5 class="modal-title">Error Details</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <pre id="errorMessageContent" class="bg-light p-3 rounded" style="max-height: 400px; overflow-y: auto;"></pre>
                </div>
            </div>
        </div>
    </div>

    <!-- <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script> -->
    <?php include_once('includes/script.php'); ?>

    <script>
        function showLogDetails(log) {
            let html = `
                <div class="row">
                    <div class="col-md-6">
                        <p><strong>ID:</strong> ${log.id}</p>
                        <p><strong>Date:</strong> ${log.created_at}</p>
                        <p><strong>User:</strong> ${log.username || 'System'} ${log.full_name ? '(' + log.full_name + ')' : ''}</p>
                        <p><strong>Action:</strong> <span class="badge bg-info">${log.action_type}</span></p>
                        <p><strong>Module:</strong> <span class="badge bg-secondary">${log.module}</span></p>
                    </div>
                    <div class="col-md-6">
                        <p><strong>Status:</strong> ${log.status === 'success' ? '<span class="badge bg-success">Success</span>' : '<span class="badge bg-danger">Error</span>'}</p>
                        <p><strong>IP Address:</strong> ${log.ip_address}</p>
                        <p><strong>User Agent:</strong> <small>${log.user_agent || 'N/A'}</small></p>
                        <p><strong>Request Method:</strong> ${log.request_method}</p>
                        <p><strong>URL:</strong> <small>${log.request_url || 'N/A'}</small></p>
                    </div>
                </div>
                
                <div class="row mt-3">
                    <div class="col-12">
                        <p><strong>Description:</strong></p>
                        <div class="alert alert-info">${log.description}</div>
                    </div>
                </div>
            `;

            if (log.error_message) {
                html += `
                    <div class="row mt-3">
                        <div class="col-12">
                            <p><strong>Error Message:</strong></p>
                            <div class="alert alert-danger">${log.error_message}</div>
                        </div>
                    </div>
                `;
            }

            if (log.request_data && log.request_data !== 'null') {
                try {
                    const requestData = JSON.parse(log.request_data);
                    html += `
                        <div class="row mt-3">
                            <div class="col-12">
                                <p><strong>Request Data:</strong></p>
                                <div class="log-data">${JSON.stringify(requestData, null, 2)}</div>
                            </div>
                        </div>
                    `;
                } catch (e) {
                    if (log.request_data && log.request_data.trim() !== '') {
                        html += `
                            <div class="row mt-3">
                                <div class="col-12">
                                    <p><strong>Request Data:</strong></p>
                                    <div class="log-data">${log.request_data}</div>
                                </div>
                            </div>
                        `;
                    }
                }
            }

            if (log.response_data && log.response_data !== 'null') {
                try {
                    const responseData = JSON.parse(log.response_data);
                    html += `
                        <div class="row mt-3">
                            <div class="col-12">
                                <p><strong>Response Data:</strong></p>
                                <div class="log-data">${JSON.stringify(responseData, null, 2)}</div>
                            </div>
                        </div>
                    `;
                } catch (e) {
                    if (log.response_data && log.response_data.trim() !== '') {
                        html += `
                            <div class="row mt-3">
                                <div class="col-12">
                                    <p><strong>Response Data:</strong></p>
                                    <div class="log-data">${log.response_data}</div>
                                </div>
                            </div>
                        `;
                    }
                }
            }

            document.getElementById('logDetailsContent').innerHTML = html;
            new bootstrap.Modal(document.getElementById('logDetailsModal')).show();
        }

        function showErrorMessage(errorMessage) {
            document.getElementById('errorMessageContent').textContent = errorMessage;
            new bootstrap.Modal(document.getElementById('errorMessageModal')).show();
        }

        function clearFilters() {
            window.location.href = 'logs.php';
        }

        function exportLogs() {
            const filters = <?= json_encode($filters) ?>;
            let url = 'export_logs.php';
            if (Object.keys(filters).length > 0) {
                url += '?' + new URLSearchParams(filters).toString();
            }
            window.open(url, '_blank');
        }

        // Initialize DataTable
        $(document).ready(function() {
            $('#logs_table').DataTable({
                paging: false,
                // searching: false,
                // info: false,
                order: [
                    
                ],
                columnDefs: [{
                        orderable: false,
                        targets: [8]
                    } // Disable sorting on Actions column
                ]
            });
        });

        $(document).ready(function () {
            $(document).on("click", "#back_home", function () {
                window.location.href="dashboard.php";
            });
        });
    </script>


</body>

</html>