<?php
// log_functions.php
require_once 'config/database.php';

class SystemLogger
{
     private $pdo;

     public function __construct($pdo)
     {
          $this->pdo = $pdo;
     }

     /**
      * Log system activity
      */
     public function log($action_type, $module, $description, $status = 'success', $error_message = null, $data = [])
     {
          try {
               $stmt = $this->pdo->prepare("
                INSERT INTO system_logs (
                    user_id, action_type, module, description, 
                    ip_address, user_agent, request_method, request_url,
                    request_data, response_data, status, error_message
                ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
            ");

               // Get user ID from session if available
               $user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : null;

               // Get request information
               $ip_address = $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
               $user_agent = $_SERVER['HTTP_USER_AGENT'] ?? '';
               $request_method = $_SERVER['REQUEST_METHOD'] ?? 'GET';
               $request_url = $_SERVER['REQUEST_URI'] ?? '';

               // Prepare data
               $request_data = !empty($_POST) ? json_encode($_POST, JSON_PRETTY_PRINT) : null;
               $response_data = !empty($data) ? json_encode($data, JSON_PRETTY_PRINT) : null;

               $stmt->execute([
                    $user_id,
                    $action_type,
                    $module,
                    $description,
                    $ip_address,
                    $user_agent,
                    $request_method,
                    $request_url,
                    $request_data,
                    $response_data,
                    $status,
                    $error_message
               ]);

               return $this->pdo->lastInsertId();
          } catch (Exception $e) {
               // Log to file if database logging fails
               error_log("Logging failed: " . $e->getMessage());
               return false;
          }
     }

     /**
      * Log user login
      */
     public function logLogin($username, $status, $error_message = null)
     {
          return $this->log(
               'login',
               'authentication',
               "User login attempt: $username",
               $status,
               $error_message,
               ['username' => $username]
          );
     }

     /**
      * Log user logout
      */
     public function logLogout($username)
     {
          return $this->log(
               'logout',
               'authentication',
               "User logout: $username",
               'success',
               null,
               ['username' => $username]
          );
     }

     /**
      * Log product operations
      */
     public function logProduct($action, $product_id, $product_name, $details = [])
     {
          return $this->log(
               $action,
               'products',
               "Product $action: $product_name (ID: $product_id)",
               'success',
               null,
               array_merge(['product_id' => $product_id, 'product_name' => $product_name], $details)
          );
     }

     /**
      * Log sales operations
      */
     public function logSale($action, $sale_id, $bill_number, $details = [])
     {
          return $this->log(
               $action,
               'sales',
               "Sale $action: $bill_number (ID: $sale_id)",
               'success',
               null,
               array_merge(['sale_id' => $sale_id, 'bill_number' => $bill_number], $details)
          );
     }

     /**
      * Log stock operations
      */
     public function logStock($action, $product_id, $change_amount, $reason, $details = [])
     {
          return $this->log(
               'stock_' . $action,
               'inventory',
               "Stock $action: Product ID $product_id, Change: $change_amount",
               'success',
               null,
               array_merge([
                    'product_id' => $product_id,
                    'change_amount' => $change_amount,
                    'reason' => $reason
               ], $details)
          );
     }

     /**
      * Log shift operations
      */
     public function logShift($action, $shift_id, $cashier_id, $details = [])
     {
          return $this->log(
               'shift_' . $action,
               'shifts',
               "Shift $action: ID $shift_id for Cashier ID $cashier_id",
               'success',
               null,
               array_merge(['shift_id' => $shift_id, 'cashier_id' => $cashier_id], $details)
          );
     }

     /**
      * Log system errors
      */
     public function logError($module, $description, $error_message)
     {
          return $this->log(
               'error',
               $module,
               $description,
               'error',
               $error_message
          );
     }


     /**
      * Get logs with filters
      */
     /**
      * Get logs with filters (COMPLETELY FIXED VERSION)
      */
     public function getLogs($filters = [], $limit = 100, $offset = 0)
     {
          $where = [];
          $params = [];

          // Build WHERE conditions - always use table aliases
          if (!empty($filters['user_id'])) {
               $where[] = "sl.user_id = ?";
               $params[] = (int)$filters['user_id'];
          }

          if (!empty($filters['action_type'])) {
               $where[] = "sl.action_type = ?";
               $params[] = $filters['action_type'];
          }

          if (!empty($filters['module'])) {
               $where[] = "sl.module = ?";
               $params[] = $filters['module'];
          }

          if (!empty($filters['status'])) {
               $where[] = "sl.status = ?";
               $params[] = $filters['status'];
          }

          if (!empty($filters['date_from'])) {
               $where[] = "DATE(sl.created_at) >= ?";
               $params[] = $filters['date_from'];
          }

          if (!empty($filters['date_to'])) {
               $where[] = "DATE(sl.created_at) <= ?";
               $params[] = $filters['date_to'];
          }

          if (!empty($filters['search'])) {
               $where[] = "(sl.description LIKE ? OR sl.request_url LIKE ?)";
               $search_term = "%" . str_replace('%', '\%', $filters['search']) . "%";
               $params[] = $search_term;
               $params[] = $search_term;
          }

          $where_clause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";

          // Cast to integers for safety
          $limit = (int)$limit;
          $offset = (int)$offset;

          $sql = "
                    SELECT 
                         sl.id, sl.user_id, sl.action_type, sl.module, 
                         sl.description, sl.ip_address, sl.user_agent,
                         sl.request_method, sl.request_url, sl.request_data,
                         sl.response_data, sl.status, sl.error_message,
                         sl.created_at,
                         u.username, u.full_name 
                    FROM system_logs sl
                    LEFT JOIN users u ON sl.user_id = u.id
                    $where_clause
                    ORDER BY sl.created_at DESC
                    LIMIT $limit OFFSET $offset
               ";

          try {
               $stmt = $this->pdo->prepare($sql);
               $stmt->execute($params);
               return $stmt->fetchAll(PDO::FETCH_ASSOC);
          } catch (Exception $e) {
               error_log("Error getting logs: " . $e->getMessage());
               return [];
          }
     }

     /**
      * Get log statistics
      */
     public function getStatistics($days = 30)
     {
          $stmt = $this->pdo->prepare("
            SELECT 
                DATE(created_at) as date,
                COUNT(*) as total_actions,
                SUM(CASE WHEN status = 'error' THEN 1 ELSE 0 END) as errors,
                COUNT(DISTINCT user_id) as unique_users
            FROM system_logs
            WHERE created_at >= DATE_SUB(NOW(), INTERVAL ? DAY)
            GROUP BY DATE(created_at)
            ORDER BY created_at DESC
        ");

          $stmt->execute([$days]);
          return $stmt->fetchAll(PDO::FETCH_ASSOC);
     }

     /**
      * Clean old logs (keep only last 90 days)
      */
     public function cleanOldLogs($days = 90)
     {
          $stmt = $this->pdo->prepare("
            DELETE FROM system_logs 
            WHERE created_at < DATE_SUB(NOW(), INTERVAL ? DAY)
        ");

          return $stmt->execute([$days]);
     }


     /**
      * Log end day stock operations
      */
     public function logEndDayStock($action, $date, $details = [])
     {
          return $this->log(
               'end_day_stock_' . $action,
               'inventory',
               "End day stock $action for date: $date",
               'success',
               null,
               array_merge(['date' => $date], $details)
          );
     }

     /**
      * Log bulk operations
      */
     public function logBulkOperation($action, $module, $total_items, $details = [])
     {
          return $this->log(
               'bulk_' . $action,
               $module,
               "Bulk $action completed for $total_items items",
               'success',
               null,
               array_merge(['total_items' => $total_items], $details)
          );
     }

     /**
      * Generate end day stock report
      */
     public function generateEndDayStockReport($date, $end_day_stock_id = null)
     {
          try {
               $where = "WHERE date = ?";
               $params = [$date];

               if ($end_day_stock_id) {
                    $where .= " AND id = ?";
                    $params[] = $end_day_stock_id;
               }

               $sql = "
            SELECT 
                eds.*,
                p.name as product_name,
                p.p_code as product_code,
                u.username as created_by_username
            FROM end_day_stock eds
            LEFT JOIN products p ON eds.product_id = p.id
            LEFT JOIN users u ON eds.created_by = u.id
            $where
            ORDER BY p.name
        ";

               $stmt = $this->pdo->prepare($sql);
               $stmt->execute($params);
               $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

               // Calculate summary statistics
               $summary = [
                    'total_products' => count($results),
                    'products_with_difference' => 0,
                    'total_positive_difference' => 0,
                    'total_negative_difference' => 0,
                    'total_absolute_difference' => 0
               ];

               foreach ($results as $row) {
                    if ($row['difference'] != 0) {
                         $summary['products_with_difference']++;
                         if ($row['difference'] > 0) {
                              $summary['total_positive_difference'] += $row['difference'];
                         } else {
                              $summary['total_negative_difference'] += abs($row['difference']);
                         }
                         $summary['total_absolute_difference'] += abs($row['difference']);
                    }
               }

               return [
                    'success' => true,
                    'data' => $results,
                    'summary' => $summary,
                    'date' => $date
               ];
          } catch (Exception $e) {
               error_log("Error generating end day stock report: " . $e->getMessage());
               return [
                    'success' => false,
                    'error' => $e->getMessage()
               ];
          }
     }
}

// Initialize logger
$logger = new SystemLogger($pdo);
