<?php
// price-function.php

// Database configuration
require_once 'config/database.php';

// Khmer categories mapping
$categories = [
    'all' => 'ទាំងអស់',
    'necklace' => 'បន្តោង',
    'bracelet' => 'ក្រវិល',
    'armband' => 'ខ្សែដៃ',
    'ring' => 'ចិញ្ចៀន',
    'chain' => 'ខ្សែក',
    'other' => 'ផ្សេងៗ'
];

/**
 * Get products without completed price (price is 0, null, or is_update = 0)
 */
function getProductsWithoutPrice($category = 'all', $pdo, $limit = 1, $offset = 0) {
    $sql = "SELECT * FROM products WHERE (price IS NULL OR price = 0 OR is_update = 0)";
    
    if ($category !== 'all') {
        // Find the Khmer category name
        global $categories;
        $khmer_category = $categories[$category] ?? $category;
        $sql .= " AND category = :category";
    }
    
    $sql .= " ORDER BY id ASC LIMIT :limit OFFSET :offset";
    
    $stmt = $pdo->prepare($sql);
    
    if ($category !== 'all') {
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
    }
    
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

/**
 * Get total count of products without price for pagination
 */
function getTotalProductsWithoutPrice($category = 'all', $pdo) {
    $sql = "SELECT COUNT(*) as total FROM products WHERE (price IS NULL OR price = 0 OR is_update = 0)";
    
    if ($category !== 'all') {
        global $categories;
        $khmer_category = $categories[$category] ?? $category;
        $sql .= " AND category = :category";
    }
    
    $stmt = $pdo->prepare($sql);
    
    if ($category !== 'all') {
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
    }
    
    $stmt->execute();
    return $stmt->fetch(PDO::FETCH_ASSOC)['total'];
}

/**
 * Update product price
 */
function updateProductPrice($product_id, $price, $stock, $pdo) {
    $sql = "UPDATE products SET price = :price, stock = :stock , is_update = 1, updated_at = NOW() WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':price', $price, PDO::PARAM_STR);
    $stmt->bindValue(':stock', $stock, PDO::PARAM_INT);
    $stmt->bindValue(':id', $product_id, PDO::PARAM_INT);
    return $stmt->execute();
}

/**
 * Get single product by ID
 */
function getProductById($product_id, $pdo) {
    $sql = "SELECT * FROM products WHERE id = :id";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':id', $product_id, PDO::PARAM_INT);
    $stmt->execute();
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

/**
 * Get next product ID without price
 */
function getNextProductId($current_id, $category = 'all', $pdo) {
    $sql = "SELECT MIN(id) as next_id FROM products 
            WHERE (price IS NULL OR price = 0 OR is_update = 0) 
            AND id > :current_id";
    
    if ($category !== 'all') {
        global $categories;
        $khmer_category = $categories[$category] ?? $category;
        $sql .= " AND category = :category";
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':current_id', $current_id, PDO::PARAM_INT);
    
    if ($category !== 'all') {
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
    }
    
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result['next_id']) {
        return $result['next_id'];
    }
    
    // If no next, get the first one
    return getFirstProductId($category, $pdo);
}

/**
 * Get previous product ID without price
 */
function getPreviousProductId($current_id, $category = 'all', $pdo) {
    $sql = "SELECT MAX(id) as prev_id FROM products 
            WHERE (price IS NULL OR price = 0 OR is_update = 0) 
            AND id < :current_id";
    
    if ($category !== 'all') {
        global $categories;
        $khmer_category = $categories[$category] ?? $category;
        $sql .= " AND category = :category";
    }
    
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':current_id', $current_id, PDO::PARAM_INT);
    
    if ($category !== 'all') {
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
    }
    
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($result['prev_id']) {
        return $result['prev_id'];
    }
    
    // If no previous, get the last one
    return getLastProductId($category, $pdo);
}

/**
 * Get first product ID without price
 */
function getFirstProductId($category = 'all', $pdo) {
    $sql = "SELECT MIN(id) as first_id FROM products 
            WHERE (price IS NULL OR price = 0 OR is_update = 0)";
    
    if ($category !== 'all') {
        global $categories;
        $khmer_category = $categories[$category] ?? $category;
        $sql .= " AND category = :category";
    }
    
    $stmt = $pdo->prepare($sql);
    
    if ($category !== 'all') {
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
    }
    
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['first_id'] ?? null;
}

/**
 * Get last product ID without price
 */
function getLastProductId($category = 'all', $pdo) {
    $sql = "SELECT MAX(id) as last_id FROM products 
            WHERE (price IS NULL OR price = 0 OR is_update = 0)";
    
    if ($category !== 'all') {
        global $categories;
        $khmer_category = $categories[$category] ?? $category;
        $sql .= " AND category = :category";
    }
    
    $stmt = $pdo->prepare($sql);
    
    if ($category !== 'all') {
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
    }
    
    $stmt->execute();
    $result = $stmt->fetch(PDO::FETCH_ASSOC);
    return $result['last_id'] ?? null;
}

/**
 * Get progress statistics - CORRECTED VERSION
 */
function getProgressStats($category = 'all', $pdo) {
    global $categories;
    
    // Initialize results array
    $results = ['total' => 0, 'done' => 0, 'pending' => 0, 'percentage' => 0];
    
    if ($category !== 'all') {
        $khmer_category = $categories[$category] ?? $category;
        
        // Build queries with proper WHERE/AND clauses
        $sql_total = "SELECT COUNT(*) as total FROM products WHERE category = :category";
        $sql_done = "SELECT COUNT(*) as done FROM products WHERE price > 0 AND is_update = 1 AND category = :category";
        $sql_pending = "SELECT COUNT(*) as pending FROM products WHERE (price IS NULL OR price = 0 OR is_update = 0) AND category = :category";
        
        // Prepare and execute total products query
        $stmt = $pdo->prepare($sql_total);
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
        $stmt->execute();
        $results['total'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Prepare and execute done products query
        $stmt = $pdo->prepare($sql_done);
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
        $stmt->execute();
        $results['done'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['done'];
        
        // Prepare and execute pending products query
        $stmt = $pdo->prepare($sql_pending);
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
        $stmt->execute();
        $results['pending'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['pending'];
        
    } else {
        // For 'all' category
        $sql_total = "SELECT COUNT(*) as total FROM products";
        $sql_done = "SELECT COUNT(*) as done FROM products WHERE price > 0 AND is_update = 1";
        $sql_pending = "SELECT COUNT(*) as pending FROM products WHERE (price IS NULL OR price = 0 OR is_update = 0)";
        
        // Prepare and execute total products query
        $stmt = $pdo->prepare($sql_total);
        $stmt->execute();
        $results['total'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Prepare and execute done products query
        $stmt = $pdo->prepare($sql_done);
        $stmt->execute();
        $results['done'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['done'];
        
        // Prepare and execute pending products query
        $stmt = $pdo->prepare($sql_pending);
        $stmt->execute();
        $results['pending'] = (int)$stmt->fetch(PDO::FETCH_ASSOC)['pending'];
    }
    
    // Calculate percentage
    if ($results['total'] > 0) {
        $results['percentage'] = round(($results['done'] / $results['total']) * 100);
    }
    
    return $results;
}

/**
 * Get all products count by category - Helper function
 */
function getAllProductsCountByCategory($category = 'all', $pdo) {
    global $categories;
    
    if ($category !== 'all') {
        $khmer_category = $categories[$category] ?? $category;
        $sql = "SELECT COUNT(*) as total FROM products WHERE category = :category";
        $stmt = $pdo->prepare($sql);
        $stmt->bindValue(':category', $khmer_category, PDO::PARAM_STR);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$result['total'];
    } else {
        $sql = "SELECT COUNT(*) as total FROM products";
        $stmt = $pdo->prepare($sql);
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return (int)$result['total'];
    }
}