let cropper = null;
let originalImageData = null;
let currentFile = null;
let isProcessing = false;

// Format file size
function formatFileSize(bytes) {
  if (bytes === 0) return "0 Bytes";
  const k = 1024;
  const sizes = ["Bytes", "KB", "MB", "GB"];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
}

// Update preview image
function updatePreview() {
  if (!cropper) return;

  const canvas = cropper.getCroppedCanvas({
    width: 200,
    height: 200,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    const preview = document.getElementById("cropPreview");
    preview.src = canvas.toDataURL("image/jpeg", 0.9);

    // Update size estimation
    updateSizeEstimation();
  }
}

// Update size estimation
function updateSizeEstimation() {
  if (!cropper) return;

  const quality =
    parseInt(document.getElementById("qualitySlider").value) / 100;
  const canvas = cropper.getCroppedCanvas({
    width: 500,
    height: 500,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    const dataUrl = canvas.toDataURL("image/jpeg", quality);
    const base64Length = dataUrl.length - (dataUrl.indexOf(",") + 1);
    const padding =
      dataUrl.charAt(dataUrl.length - 2) === "="
        ? dataUrl.charAt(dataUrl.length - 1) === "="
          ? 2
          : 1
        : 0;
    const fileSize = (base64Length * 3) / 4 - padding;
    document.getElementById("estimatedSize").textContent =
      formatFileSize(fileSize);
  }
}

// Reset all adjustments
function resetAllAdjustments() {
  // Reset sliders
  document.getElementById("zoomSlider").value = 1;
  document.getElementById("zoomValue").textContent = "100%";

  document.getElementById("rotationSlider").value = 0;
  document.getElementById("rotationValue").textContent = "0°";

  document.getElementById("qualitySlider").value = 85;
  document.getElementById("qualityValue").textContent = "85%";

  // Reset cropper
  if (cropper) {
    cropper.reset();
    cropper.zoomTo(1);
    cropper.rotateTo(0);
  }

  // Update preview
  updatePreview();
  updateSizeEstimation();
}

// Show loading
function showLoading() {
  document.getElementById("loadingOverlay").classList.remove("d-none");
  isProcessing = true;
}

// Hide loading
function hideLoading() {
  document.getElementById("loadingOverlay").classList.add("d-none");
  isProcessing = false;
}

// Apply crop and show final result
function applyCropAndShowFinal() {
  if (!cropper || isProcessing) return;

  showLoading();

  setTimeout(() => {
    const quality =
      parseInt(document.getElementById("qualitySlider").value) / 100;

    // Get cropped canvas
    const canvas = cropper.getCroppedCanvas({
      width: 800,
      height: 800,
      fillColor: "#fff",
      imageSmoothingEnabled: true,
      imageSmoothingQuality: "high",
    });

    if (canvas) {
      // Convert to base64
      const base64Image = canvas.toDataURL("image/jpeg", quality);

      // Set the base64 data
      document.getElementById("croppedImageData").value = base64Image;

      // Show final preview
      const finalPreview = document.getElementById("finalPreview");
      finalPreview.src = base64Image;
      finalPreview.onload = () => {
        // Switch to final section
        document.getElementById("editorSection").classList.add("d-none");
        document.getElementById("finalSection").classList.remove("d-none");

        // Enable final submit button
        const updateBtn = document.getElementById("updatePhotoBtn");
        updateBtn.disabled = false;
        updateBtn.innerHTML = '<i class="fas fa-save"></i> Update Photo';

        hideLoading();
      };
    } else {
      hideLoading();
      alert("Error processing image. Please try again.");
    }
  }, 300);
}

// Reset modal to initial state
function resetModal() {
  // Reset all states
  document.getElementById("dropArea").style.display = "block";
  document.getElementById("fileInfo").classList.add("d-none");
  document.getElementById("editorSection").classList.add("d-none");
  document.getElementById("finalSection").classList.add("d-none");

  // Reset form
  document.getElementById("profilePhotoInput").value = "";
  document.getElementById("croppedImageData").value = "";

  // Reset preview images
  document.getElementById("cropPreview").src = "";
  document.getElementById("finalPreview").src = "";
  document.getElementById("imageToCrop").src = "";

  // Reset cropper
  if (cropper) {
    cropper.destroy();
    cropper = null;
  }

  // Reset adjustments
  resetAllAdjustments();

  // Reset update button
  document.getElementById("updatePhotoBtn").disabled = true;
  document.getElementById("updatePhotoBtn").innerHTML =
    '<i class="fas fa-save"></i> Update Photo';

  // Reset estimated size
  document.getElementById("estimatedSize").textContent = "-";
}

// Close modal
function closeModal() {
  document.getElementById("changePhotoModal").classList.remove("show");
  document.getElementById("changePhotoModal").style.display = "none";
  document.body.classList.remove("modal-open");
  const backdrop = document.querySelector(".modal-backdrop");
  if (backdrop) backdrop.remove();
  resetModal();
}

// Open modal
function openModal() {
  resetModal();
  document.getElementById("changePhotoModal").classList.add("show");
  document.getElementById("changePhotoModal").style.display = "block";
  document.body.classList.add("modal-open");

  // Add backdrop
  const backdrop = document.createElement("div");
  backdrop.className = "modal-backdrop fade show";
  document.body.appendChild(backdrop);
}

// Initialize when DOM is ready
document.addEventListener("DOMContentLoaded", function () {
  // Elements
  const dropArea = document.getElementById("dropArea");
  const browseBtn = document.getElementById("browseBtn");
  const profilePhotoInput = document.getElementById("profilePhotoInput");
  const changeFileBtn = document.getElementById("changeFileBtn");
  const fileInfo = document.getElementById("fileInfo");
  const modal = document.getElementById("changePhotoModal");

  // Handle files
  function handleFiles(files) {
    if (!files || files.length === 0) return;

    const file = files[0];

    // Validate file type
    if (!file.type.match("image.*")) {
      alert("Please select an image file (JPG, PNG, GIF, etc.)");
      return;
    }

    // Validate file size (max 10MB)
    if (file.size > 10 * 1024 * 1024) {
      alert("File size too large. Maximum size is 10MB.");
      return;
    }

    currentFile = file;

    // Show file info
    document.getElementById("fileName").textContent = file.name;
    document.getElementById("fileSize").textContent = formatFileSize(file.size);
    fileInfo.classList.remove("d-none");

    // Start editing
    startImageEditing(file);
  }

  function startImageEditing(file) {
    const reader = new FileReader();

    reader.onload = function (e) {
      // Hide upload section, show editor
      dropArea.style.display = "none";
      fileInfo.classList.add("d-none");
      document.getElementById("editorSection").classList.remove("d-none");

      // Initialize image for cropping
      const image = document.getElementById("imageToCrop");
      image.onload = function () {
        // Destroy existing cropper
        if (cropper) {
          cropper.destroy();
        }

        // Initialize cropper
        cropper = new Cropper(image, {
          aspectRatio: 1,
          viewMode: 2,
          autoCropArea: 0.8,
          responsive: true,
          restore: true,
          guides: true,
          center: true,
          highlight: false,
          cropBoxMovable: true,
          cropBoxResizable: true,
          background: false,
          modal: false,
          toggleDragModeOnDblclick: false,
          ready: function () {
            // Update preview
            updatePreview();
            updateSizeEstimation();

            // Enable update button
            document.getElementById("updatePhotoBtn").disabled = false;
          },
          crop: function () {
            updatePreview();
          },
        });

        // Initialize adjustments
        initAdjustments();
      };

      // Set src after onload handler is set
      image.src = e.target.result;

      // Store original data
      originalImageData = e.target.result;
    };

    reader.onerror = function () {
      alert("Error reading file. Please try again.");
      resetModal();
    };

    reader.readAsDataURL(file);
  }

  function initAdjustments() {
    // Zoom slider
    const zoomSlider = document.getElementById("zoomSlider");
    const zoomValue = document.getElementById("zoomValue");

    zoomSlider.addEventListener("input", function () {
      const value = parseFloat(this.value);
      zoomValue.textContent = Math.round(value * 100) + "%";
      if (cropper) cropper.zoomTo(value);
    });

    // Zoom buttons
    document.getElementById("zoomIn").addEventListener("click", function () {
      if (cropper) {
        cropper.zoom(0.1);
        const currentZoom = cropper.getData().zoom;
        zoomSlider.value = currentZoom;
        zoomValue.textContent = Math.round(currentZoom * 100) + "%";
      }
    });

    document.getElementById("zoomOut").addEventListener("click", function () {
      if (cropper) {
        cropper.zoom(-0.1);
        const currentZoom = cropper.getData().zoom;
        zoomSlider.value = currentZoom;
        zoomValue.textContent = Math.round(currentZoom * 100) + "%";
      }
    });

    // Rotation controls
    const rotationSlider = document.getElementById("rotationSlider");
    const rotationValue = document.getElementById("rotationValue");

    rotationSlider.addEventListener("input", function () {
      const value = parseInt(this.value);
      rotationValue.textContent = value + "°";
      if (cropper) cropper.rotateTo(value);
    });

    document
      .getElementById("rotateLeft")
      .addEventListener("click", function () {
        if (cropper) {
          const current = parseInt(rotationSlider.value);
          const newValue = current - 15;
          rotationSlider.value = newValue;
          rotationValue.textContent = newValue + "°";
          cropper.rotateTo(newValue);
        }
      });

    document
      .getElementById("rotateRight")
      .addEventListener("click", function () {
        if (cropper) {
          const current = parseInt(rotationSlider.value);
          const newValue = current + 15;
          rotationSlider.value = newValue;
          rotationValue.textContent = newValue + "°";
          cropper.rotateTo(newValue);
        }
      });

    // Quality slider
    const qualitySlider = document.getElementById("qualitySlider");
    const qualityValue = document.getElementById("qualityValue");

    qualitySlider.addEventListener("input", function () {
      const value = parseInt(this.value);
      qualityValue.textContent = value + "%";
      updateSizeEstimation();
    });

    // Reset button
    document
      .getElementById("resetBtn")
      .addEventListener("click", resetAllAdjustments);

    // Cancel crop button
    document
      .getElementById("cancelCropBtn")
      .addEventListener("click", function () {
        resetModal();
        document.getElementById("dropArea").style.display = "block";
        fileInfo.classList.add("d-none");
        document.getElementById("editorSection").classList.add("d-none");
      });

    // Apply crop button
    document
      .getElementById("applyCropBtn")
      .addEventListener("click", applyCropAndShowFinal);

    // Edit again button
    document
      .getElementById("editAgainBtn")
      .addEventListener("click", function () {
        document.getElementById("finalSection").classList.add("d-none");
        document.getElementById("editorSection").classList.remove("d-none");
      });
  }

  // Drag & Drop functionality
  function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
  }

  function highlight() {
    dropArea.style.borderColor = "#007bff";
    dropArea.style.background = "#e6f7ff";
    dropArea.style.transform = "scale(1.02)";
  }

  function unhighlight() {
    dropArea.style.borderColor = "#ced4da";
    dropArea.style.background = "#f8f9fa";
    dropArea.style.transform = "scale(1)";
  }

  // Drag events
  ["dragenter", "dragover"].forEach((eventName) => {
    dropArea.addEventListener(
      eventName,
      function (e) {
        preventDefaults(e);
        highlight();
      },
      false
    );
  });

  ["dragleave", "drop"].forEach((eventName) => {
    dropArea.addEventListener(
      eventName,
      function (e) {
        preventDefaults(e);
        unhighlight();
      },
      false
    );
  });

  // Handle drop
  dropArea.addEventListener(
    "drop",
    function (e) {
      const dt = e.dataTransfer;
      const files = dt.files;
      handleFiles(files);
    },
    false
  );

  // Click events
  dropArea.addEventListener("click", function (e) {
    if (e.target !== browseBtn) {
      profilePhotoInput.click();
    }
  });

  browseBtn.addEventListener("click", function (e) {
    e.stopPropagation();
    profilePhotoInput.click();
  });

  // File input change
  profilePhotoInput.addEventListener("change", function (e) {
    handleFiles(this.files);
  });

  // Change file button
  changeFileBtn.addEventListener("click", function () {
    profilePhotoInput.click();
  });

  // Form submission
  // Form submission
  document
    .getElementById("changePhotoForm")
    .addEventListener("submit", function (e) {
      e.preventDefault();

      if (isProcessing) {
        alert("Please wait while the image is being processed.");
        return false;
      }

      const imageData = document.getElementById("croppedImageData").value;
      if (!imageData || imageData.indexOf("data:image/") !== 0) {
        alert("Please select and process an image first!");
        return false;
      }

      showLoading();

      // Create form data
      const formData = new FormData();
      formData.append("cropped_image_data", imageData);
      formData.append("change_profile", "1");

      // Submit via traditional form POST (not fetch)
      // Create a hidden iframe for submission
      const iframe = document.createElement("iframe");
      iframe.name = "uploadFrame";
      iframe.style.display = "none";
      document.body.appendChild(iframe);

      // Create a form for traditional submission
      const tempForm = document.createElement("form");
      tempForm.method = "POST";
      tempForm.action = ""; // Submit to same page
      tempForm.target = "uploadFrame";
      tempForm.style.display = "none";

      const input1 = document.createElement("input");
      input1.type = "hidden";
      input1.name = "cropped_image_data";
      input1.value = imageData;

      const input2 = document.createElement("input");
      input2.type = "hidden";
      input2.name = "change_profile";
      input2.value = "1";

      tempForm.appendChild(input1);
      tempForm.appendChild(input2);
      document.body.appendChild(tempForm);

      // Handle response from iframe
      iframe.onload = function () {
        hideLoading();
        try {
          const response = iframe.contentDocument.body.innerHTML;
          console.log("Response:", response);

          // Check if response contains success message
          if (response.includes("successfully")) {
            // alert("Profile picture updated successfully!");
            closeModal();
            window.location.reload();
          } else if (response.includes("alert")) {
            // Extract alert message
            const match = response.match(/alert\('([^']+)'/);
            if (match) {
              alert(match[1]);
            }
          }
        } catch (e) {
          console.error("Error reading response:", e);
        }

        // Clean up
        document.body.removeChild(tempForm);
        document.body.removeChild(iframe);
      };

      tempForm.submit();

      return false;
    });

  // Close modal events
  const closeButtons = document.querySelectorAll(
    '#changePhotoModal .close, #changePhotoModal .btn-secondary[data-dismiss="modal"]'
  );
  closeButtons.forEach((button) => {
    button.addEventListener("click", closeModal);
  });

  // Close modal on outside click
  window.addEventListener("click", function (event) {
    if (event.target === modal) {
      closeModal();
    }
  });

  // Keyboard events
  document.addEventListener("keydown", function (e) {
    if (e.key === "Escape" && modal.classList.contains("show")) {
      closeModal();
    }
  });

  // Make openModal function globally available
  window.openPhotoModal = openModal;
});
