<?php
date_default_timezone_set('Asia/Phnom_Penh');
// Start session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include database connection
require_once 'config/database.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['user_role'], ['admin', 'owner'])) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Get shift ID from request
if (!isset($_GET['id'])) {
    echo json_encode(['success' => false, 'error' => 'Shift ID required']);
    exit;
}

$shift_id = $_GET['id'];

try {
    // Fetch shift details
    $shift_stmt = $pdo->prepare("
        SELECT s.*, u.username, u.full_name 
        FROM shifts s 
        JOIN users u ON s.cashier_id = u.id 
        WHERE s.id = ?
    ");
    $shift_stmt->execute([$shift_id]);
    $shift = $shift_stmt->fetch();

    if (!$shift) {
        echo json_encode(['success' => false, 'error' => 'Shift not found']);
        exit;
    }

    // Fetch shift sales
    $sales_stmt = $pdo->prepare("
        SELECT s.*, COUNT(si.id) as item_count
        FROM sales s 
        LEFT JOIN sale_items si ON s.id = si.sale_id
        WHERE s.shift_id = ?
        GROUP BY s.id
        ORDER BY s.created_at DESC
    ");
    $sales_stmt->execute([$shift_id]);
    $sales = $sales_stmt->fetchAll();

    // Fetch top items for this shift
    $top_items_stmt = $pdo->prepare("
        SELECT p.name, SUM(si.quantity) as qty_sold, SUM(si.total_price) as revenue
        FROM sale_items si
        JOIN sales s ON si.sale_id = s.id
        JOIN products p ON si.product_id = p.id
        WHERE s.shift_id = ?
        GROUP BY p.id, p.name
        ORDER BY qty_sold DESC
        LIMIT 10
    ");
    $top_items_stmt->execute([$shift_id]);
    $top_items = $top_items_stmt->fetchAll();

    // Calculate shift duration
    $duration = time() - strtotime($shift['start_time']);
    $hours = floor($duration / 3600);
    $minutes = floor(($duration % 3600) / 60);

    // Generate HTML content
    $html = "
        <div class='shift-details'>
            <div class='shift-info'>
                <h4>Shift Information</h4>
                <p><strong>Cashier:</strong> {$shift['full_name']} ({$shift['username']})</p>
                <p><strong>Start Time:</strong> " . date('d/M/Y H:i', strtotime($shift['start_time'])) . "</p>
                <p><strong>Duration:</strong> {$hours}h {$minutes}m</p>
                <p><strong>Total Sales:</strong> $" . number_format($shift['total_sales'], 2) . "</p>
                <p><strong>Total Items:</strong> {$shift['total_items']}</p>
            </div>

            <div class='shift-sales'>
                <h4>Sales in This Shift</h4>
    ";
    
    if (count($sales) > 0) {
        $html .= "
            <table class='data-table'>
                <thead>
                    <tr>
                        <th>Bill No</th>
                        <th>Time</th>
                        <th>Items</th>
                        <th>Total</th>
                    </tr>
                </thead>
                <tbody>
        ";
        
        foreach ($sales as $sale) {
            $html .= "
                <tr>
                    <td>{$sale['bill_number']}</td>
                    <td>" . date('H:i', strtotime($sale['created_at'])) . "</td>
                    <td>{$sale['item_count']}</td>
                    <td>$" . number_format($sale['total_amount'], 2) . "</td>
                </tr>
            ";
        }
        
        $html .= "
                </tbody>
            </table>
        ";
    } else {
        $html .= "<p>No sales recorded in this shift.</p>";
    }
    
    $html .= "
            </div>

            <div class='top-items'>
                <h4>Top Selling Items</h4>
    ";
    
    if (count($top_items) > 0) {
        $html .= "
            <table class='data-table'>
                <thead>
                    <tr>
                        <th>Product</th>
                        <th>Quantity Sold</th>
                        <th>Revenue</th>
                    </tr>
                </thead>
                <tbody>
        ";
        
        foreach ($top_items as $item) {
            $html .= "
                <tr>
                    <td>{$item['name']}</td>
                    <td>{$item['qty_sold']}</td>
                    <td>$" . number_format($item['revenue'], 2) . "</td>
                </tr>
            ";
        }
        
        $html .= "
                </tbody>
            </table>
        ";
    } else {
        $html .= "<p>No items sold in this shift.</p>";
    }
    
    $html .= "
            </div>
        </div>
    ";

    echo json_encode(['success' => true, 'html' => $html]);
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
}
?>