<?php
date_default_timezone_set('Asia/Phnom_Penh');
// Start session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// Include database connection
require_once 'config/database.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if user is logged in and is admin
if (!isset($_SESSION['user_id']) || !in_array($_SESSION['user_role'], ['admin', 'owner'])) {
    echo json_encode(['success' => false, 'error' => 'Unauthorized']);
    exit;
}

// Get sale ID from request
if (!isset($_GET['id'])) {
    echo json_encode(['success' => false, 'error' => 'Sale ID required']);
    exit;
}

$sale_id = $_GET['id'];
$action = $_GET['action'] ?? 'view';

try {
    // Fetch sale details
    $sale_stmt = $pdo->prepare("
        SELECT s.*, u.username, u.full_name 
        FROM sales s 
        JOIN users u ON s.cashier_id = u.id 
        WHERE s.id = ?
    ");
    $sale_stmt->execute([$sale_id]);
    $sale = $sale_stmt->fetch();

    if (!$sale) {
        echo json_encode(['success' => false, 'error' => 'Sale not found']);
        exit;
    }

    // Fetch sale items
    $items_stmt = $pdo->prepare("
        SELECT si.*, p.name, p.stock
        FROM sale_items si
        JOIN products p ON si.product_id = p.id
        WHERE si.sale_id = ?
    ");
    $items_stmt->execute([$sale_id]);
    $items = $items_stmt->fetchAll();

    if ($action === 'edit') {
        // Edit mode - show items with checkboxes for removal
        $html = "
            <div class='bill-info'>
                <p><strong>Bill No:</strong> {$sale['bill_number']}</p>
                <p><strong>Date:</strong> " . date('d/M/Y H:i', strtotime($sale['created_at'])) . "</p>
                <p><strong>Cashier:</strong> {$sale['full_name']}</p>
                <p><strong>Total Amount:</strong> $" . number_format($sale['total_amount'], 2) . "</p>
            </div>

            <div class='bill-items-edit'>
                <h4>Select items to remove (stock will be restored):</h4>
                <div class='bill-items-list'>
        ";
        
        foreach ($items as $item) {
            $html .= "
                <div class='bill-item-row'>
                    <div>
                        <input type='checkbox' name='items_to_remove[]' value='{$item['id']}' id='item_{$item['id']}'>
                        <label for='item_{$item['id']}'><strong>{$item['name']}</strong></label>
                    </div>
                    <div>
                        <span>Qty: {$item['quantity']} × $" . number_format($item['unit_price'], 2) . " = $" . number_format($item['total_price'], 2) . "</span>
                    </div>
                </div>
            ";
        }
        
        $html .= "
                </div>
            </div>
        ";
        
    } else {
        // View mode - show bill details
        $html = "
            <div class='bill-info'>
                <p><strong>Bill No:</strong> {$sale['bill_number']}</p>
                <p><strong>Date:</strong> " . date('d/M/Y H:i', strtotime($sale['created_at'])) . "</p>
                <p><strong>Cashier:</strong> {$sale['full_name']}</p>
            </div>

            <div class='bill-items-view'>
                <table class='data-table'>
                    <thead>
                        <tr>
                            <th>Item</th>
                            <th>Quantity</th>
                            <th>Unit Price</th>
                            <th>Total</th>
                        </tr>
                    </thead>
                    <tbody>
        ";
        
        foreach ($items as $item) {
            $html .= "
                <tr>
                    <td>{$item['name']}</td>
                    <td>{$item['quantity']}</td>
                    <td>$" . number_format($item['unit_price'], 2) . "</td>
                    <td>$" . number_format($item['total_price'], 2) . "</td>
                </tr>
            ";
        }
        
        $html .= "
                    </tbody>
                    <tfoot>
                        <tr>
                            <td colspan='3' style='text-align: right;'><strong>Total:</strong></td>
                            <td><strong>$" . number_format($sale['total_amount'], 2) . "</strong></td>
                        </tr>
                    </tfoot>
                </table>
            </div>
        ";
    }

    echo json_encode(['success' => true, 'html' => $html]);
    
} catch (Exception $e) {
    echo json_encode(['success' => false, 'error' => 'Database error: ' . $e->getMessage()]);
}
?>