let productCropper = null;
let originalProductImage = null;
let currentProductFile = null;

// Format file size
function formatFileSize(bytes) {
  if (bytes === 0) return "0 Bytes";
  const k = 1024;
  const sizes = ["Bytes", "KB", "MB", "GB"];
  const i = Math.floor(Math.log(bytes) / Math.log(k));
  return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + " " + sizes[i];
}

// Update preview image
function updateProductPreview() {
  if (!productCropper) return;

  const canvas = productCropper.getCroppedCanvas({
    width: 300,
    height: 300,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    const preview = document.getElementById("productCropPreview");
    const quality =
      parseInt(document.getElementById("productQualitySlider").value) / 100;
    preview.src = canvas.toDataURL("image/jpeg", quality);
    updateProductSizeEstimation();
  }
}

// Update size estimation
function updateProductSizeEstimation() {
  if (!productCropper) return;

  const quality =
    parseInt(document.getElementById("productQualitySlider").value) / 100;
  const canvas = productCropper.getCroppedCanvas({
    width: 800,
    height: 800,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    const dataUrl = canvas.toDataURL("image/jpeg", quality);
    const base64Length = dataUrl.length - (dataUrl.indexOf(",") + 1);
    const padding =
      dataUrl.charAt(dataUrl.length - 2) === "="
        ? dataUrl.charAt(dataUrl.length - 1) === "="
          ? 2
          : 1
        : 0;
    const fileSize = (base64Length * 3) / 4 - padding;
    document.getElementById("productEstimatedSize").textContent =
      formatFileSize(fileSize);
  }
}

// Reset all adjustments
function resetProductAdjustments() {
  // Reset sliders
  document.getElementById("productZoomSlider").value = 1;
  document.getElementById("productZoomValue").textContent = "100%";
  document.getElementById("productRotationSlider").value = 0;
  document.getElementById("productRotationValue").textContent = "0°";
  document.getElementById("productQualitySlider").value = 85;
  document.getElementById("productQualityValue").textContent = "85%";

  // Reset cropper
  if (productCropper) {
    productCropper.reset();
    productCropper.zoomTo(1);
    productCropper.rotateTo(0);
  }

  updateProductPreview();
}

// Initialize product image editor
function initProductImageEditor(imageSrc) {
  const image = document.getElementById("productImageToCrop");
  image.src = imageSrc;

  image.onload = function () {
    // Destroy existing cropper
    if (productCropper) {
      productCropper.destroy();
    }

    // Initialize cropper
    productCropper = new Cropper(image, {
      aspectRatio: 1,
      viewMode: 1,
      autoCropArea: 0.8,
      responsive: true,
      restore: true,
      guides: true,
      center: true,
      highlight: false,
      cropBoxMovable: true,
      cropBoxResizable: true,
      background: false,
      modal: false,
      toggleDragModeOnDblclick: false,
      ready: function () {
        updateProductPreview();
        updateProductSizeEstimation();
      },
      crop: function () {
        updateProductPreview();
      },
    });

    initProductAdjustmentControls();
  };
}

// Initialize adjustment controls
function initProductAdjustmentControls() {
  // Zoom slider
  const zoomSlider = document.getElementById("productZoomSlider");
  const zoomValue = document.getElementById("productZoomValue");

  zoomSlider.addEventListener("input", function () {
    const value = parseFloat(this.value);
    zoomValue.textContent = Math.round(value * 100) + "%";
    if (productCropper) productCropper.zoomTo(value);
  });

  // Zoom buttons
  document
    .getElementById("productZoomIn")
    .addEventListener("click", function () {
      if (productCropper) {
        productCropper.zoom(0.1);
        const currentZoom = productCropper.getData().zoom;
        zoomSlider.value = currentZoom;
        zoomValue.textContent = Math.round(currentZoom * 100) + "%";
      }
    });

  document
    .getElementById("productZoomOut")
    .addEventListener("click", function () {
      if (productCropper) {
        productCropper.zoom(-0.1);
        const currentZoom = productCropper.getData().zoom;
        zoomSlider.value = currentZoom;
        zoomValue.textContent = Math.round(currentZoom * 100) + "%";
      }
    });

  // Rotation controls
  const rotationSlider = document.getElementById("productRotationSlider");
  const rotationValue = document.getElementById("productRotationValue");

  rotationSlider.addEventListener("input", function () {
    const value = parseInt(this.value);
    rotationValue.textContent = value + "°";
    if (productCropper) productCropper.rotateTo(value);
  });

  document
    .getElementById("productRotateLeft")
    .addEventListener("click", function () {
      if (productCropper) {
        const current = parseInt(rotationSlider.value);
        const newValue = current - 15;
        rotationSlider.value = newValue;
        rotationValue.textContent = newValue + "°";
        productCropper.rotateTo(newValue);
      }
    });

  document
    .getElementById("productRotateRight")
    .addEventListener("click", function () {
      if (productCropper) {
        const current = parseInt(rotationSlider.value);
        const newValue = current + 15;
        rotationSlider.value = newValue;
        rotationValue.textContent = newValue + "°";
        productCropper.rotateTo(newValue);
      }
    });

  // Quality slider
  const qualitySlider = document.getElementById("productQualitySlider");
  const qualityValue = document.getElementById("productQualityValue");

  qualitySlider.addEventListener("input", function () {
    const value = parseInt(this.value);
    qualityValue.textContent = value + "%";
    updateProductPreview();
  });
}

// Handle file selection
function handleProductFileSelect(file) {
  if (!file || !file.type.match("image.*")) {
    alert("Please select an image file (JPG, PNG, GIF, etc.)");
    return;
  }

  if (file.size > 10 * 1024 * 1024) {
    alert("File size too large. Maximum size is 10MB.");
    return;
  }

  currentProductFile = file;

  // Show file info
  document.getElementById("productFileName").textContent = file.name;
  document.getElementById("productFileSize").textContent = formatFileSize(
    file.size
  );
  document.getElementById("productFileInfo").classList.remove("d-none");

  // Show editor section
  document.getElementById("productEditorSection").classList.remove("d-none");
  document.getElementById("productFinalPreview").classList.add("d-none");

  // Read file and initialize editor
  const reader = new FileReader();
  reader.onload = function (e) {
    initProductImageEditor(e.target.result);
  };
  reader.readAsDataURL(file);
}

// Apply crop and show final result
function applyProductCrop() {
  if (!productCropper) return;

  const quality =
    parseInt(document.getElementById("productQualitySlider").value) / 100;

  // Get cropped canvas
  const canvas = productCropper.getCroppedCanvas({
    width: 800,
    height: 800,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    // Convert to base64
    const base64Image = canvas.toDataURL("image/jpeg", quality);

    // Set the base64 data in hidden field
    document.getElementById("productCroppedImageData").value = base64Image;

    // Show final preview
    const finalPreview = document.getElementById("productFinalImage");
    finalPreview.src = base64Image;

    // Switch to final preview
    document.getElementById("productEditorSection").classList.add("d-none");
    document.getElementById("productFinalPreview").classList.remove("d-none");
    document.getElementById("productFileInfo").classList.add("d-none");
  }
}

// Reset product image upload
function resetProductImageUpload() {
  document.getElementById("productDropArea").style.display = "block";
  document.getElementById("productFileInfo").classList.add("d-none");
  document.getElementById("productEditorSection").classList.add("d-none");
  document.getElementById("productFinalPreview").classList.add("d-none");
  document.getElementById("productImageInput").value = "";
  document.getElementById("productCroppedImageData").value = "";

  if (productCropper) {
    productCropper.destroy();
    productCropper = null;
  }

  resetProductAdjustments();
}

// Initialize when DOM is ready
document.addEventListener("DOMContentLoaded", function () {
  // Elements
  const dropArea = document.getElementById("productDropArea");
  const browseBtn = document.getElementById("productBrowseBtn");
  const imageInput = document.getElementById("productImageInput");
  const changeFileBtn = document.getElementById("productChangeFileBtn");
  const cancelCropBtn = document.getElementById("productCancelCropBtn");
  const applyCropBtn = document.getElementById("productApplyCropBtn");
  const resetBtn = document.getElementById("productResetBtn");
  const editAgainBtn = document.getElementById("productEditAgainBtn");

  // Drag & Drop functionality
  function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
  }

  function highlight() {
    dropArea.style.borderColor = "#007bff";
    dropArea.style.background = "#e6f7ff";
    dropArea.style.transform = "scale(1.02)";
  }

  function unhighlight() {
    dropArea.style.borderColor = "#ced4da";
    dropArea.style.background = "#f8f9fa";
    dropArea.style.transform = "scale(1)";
  }

  // Drag events
  ["dragenter", "dragover"].forEach((eventName) => {
    dropArea.addEventListener(
      eventName,
      function (e) {
        preventDefaults(e);
        highlight();
      },
      false
    );
  });

  ["dragleave", "drop"].forEach((eventName) => {
    dropArea.addEventListener(
      eventName,
      function (e) {
        preventDefaults(e);
        unhighlight();
      },
      false
    );
  });

  // Handle drop
  dropArea.addEventListener(
    "drop",
    function (e) {
      const files = e.dataTransfer.files;
      if (files.length > 0) {
        handleProductFileSelect(files[0]);
      }
    },
    false
  );

  // Click events
  dropArea.addEventListener("click", function (e) {
    if (e.target !== browseBtn) {
      imageInput.click();
    }
  });

  browseBtn.addEventListener("click", function (e) {
    e.stopPropagation();
    imageInput.click();
  });

  // File input change
  imageInput.addEventListener("change", function (e) {
    if (this.files.length > 0) {
      handleProductFileSelect(this.files[0]);
    }
  });

  // Change file button
  changeFileBtn.addEventListener("click", function () {
    resetProductImageUpload();
  });

  // Cancel crop button
  cancelCropBtn.addEventListener("click", function () {
    resetProductImageUpload();
  });

  // Apply crop button
  applyCropBtn.addEventListener("click", applyProductCrop);

  // Reset button
  resetBtn.addEventListener("click", resetProductAdjustments);

  // Edit again button
  editAgainBtn.addEventListener("click", function () {
    document.getElementById("productFinalPreview").classList.add("d-none");
    document.getElementById("productEditorSection").classList.remove("d-none");
  });

  // Close modal buttons
  document.querySelectorAll(".close, .close-modal").forEach((btn) => {
    btn.addEventListener("click", function () {
      document.getElementById("addProductModal").style.display = "none";
      resetProductImageUpload();
    });
  });
});

// --------- update1 ---------

// Variables for edit product modal
let editProductCropper = null;
let editCurrentProductFile = null;
let currentProductStock = 0;

// Main function to edit product
function editProductButton(productId) {
  // alert("Editing product ID: " + productId);
  // Reset edit modal first
  resetEditProductImageUpload();

  // Show modal
  document.getElementById("editProductModal").style.display = "block";
  // document.getElementsByTagName("form").reset();
  // Clear previous data
  document.getElementById("p_name").value = "";
  document.getElementById("p_description").value = "";
  document.getElementById("sale_price").value = "";
  document.getElementById("p_min_stock").value = "";
  document.getElementById("preview_p_img").src = "";
  document.getElementById("current_stock_value").value="";

  // Send AJAX request to fetch product data
  const xhr = new XMLHttpRequest();
  xhr.open("POST", "ajax.php", true);
  xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");

  xhr.onreadystatechange = function () {
    if (xhr.readyState === 4) {
      if (xhr.status === 200) {
        try {
          const response = JSON.parse(xhr.responseText);

          if (response.success) {
            const product = response.data;

            // Fill form with product data
            document.getElementById("edit_product_id").value = product.id;
            document.getElementById("p_name").value = product.name;
            document.getElementById("p_description").value =
              product.description || "";
            document.getElementById("sale_price").value = product.price;
            document.getElementById("p_min_stock").value = product.min_stock;
            document.getElementById("edit_current_image").value =
              product.image || "";

            // Display current stock
            currentProductStock = product.stock;
            document.getElementById("current_stock_display").textContent =
              product.stock;

            // Set category
            const categorySelect = document.getElementById("p_category");
            for (let i = 0; i < categorySelect.options.length; i++) {
              if (categorySelect.options[i].value === product.category) {
                categorySelect.options[i].selected = true;
                break;
              }
            }

            // Show current product image
            const imgPreview = document.getElementById("preview_p_img");
            if (product.image && product.image !== "default-product.jpg") {
              imgPreview.src = "assets/images/" + product.image;
              imgPreview.onerror = function () {
                this.src = "assets/images/default-product.jpg";
              };
            } else {
              imgPreview.src = "assets/images/default-product.jpg";
            }
          } else {
            alert("Error: " + response.message);
          }
        } catch (e) {
          console.error("Error parsing JSON:", e);
          alert("Error loading product data");
        }
      } else {
        alert("Server error: " + xhr.status);
      }
    }
  };

  // Send the request
  xhr.send("action=get_product&product_id=" + productId);
}

// Close edit modal
function closeEditModal() {
  document.getElementById("editProductModal").style.display = "none";
  document.body.classList.remove("modal-open");
  resetEditProductImageUpload();
}

// Reset edit product image upload
function resetEditProductImageUpload() {
  // Show current image preview
  document.getElementById("editCurrentImagePreview").style.display = "block";

  // Hide other sections
  document.getElementById("editProductDropArea").style.display = "none";
  document.getElementById("editProductFileInfo").style.display = "none";
  document.getElementById("editProductEditorSection").style.display = "none";
  document.getElementById("editProductNewPreview").style.display = "none";

  // Reset inputs
  document.getElementById("editProductImageInput").value = "";
  document.getElementById("editProductCroppedImageData").value = "";

  // Reset preview images
  document.getElementById("editProductCropPreview").src = "";
  document.getElementById("editProductNewImage").src = "";

  // Destroy cropper
  if (editProductCropper) {
    editProductCropper.destroy();
    editProductCropper = null;
  }

  // Reset adjustments
  resetEditProductAdjustments();
}

// Reset edit product adjustments
function resetEditProductAdjustments() {
  if (document.getElementById("editProductZoomSlider")) {
    document.getElementById("editProductZoomSlider").value = 1;
    document.getElementById("editProductZoomValue").textContent = "100%";
    document.getElementById("editProductRotationSlider").value = 0;
    document.getElementById("editProductRotationValue").textContent = "0°";
    document.getElementById("editProductQualitySlider").value = 85;
    document.getElementById("editProductQualityValue").textContent = "85%";
    document.getElementById("editProductEstimatedSize").textContent = "-";
  }
}

// Handle edit product file select
function handleEditProductFileSelect(file) {
  if (!file || !file.type.match("image.*")) {
    alert("Please select an image file (JPG, PNG, GIF, etc.)");
    return;
  }

  if (file.size > 10 * 1024 * 1024) {
    alert("File size too large. Maximum size is 10MB.");
    return;
  }

  editCurrentProductFile = file;

  // Show file info
  document.getElementById("editProductFileName").textContent = file.name;
  document.getElementById("editProductFileSize").textContent = formatFileSize(
    file.size
  );
  document.getElementById("editProductFileInfo").style.display = "flex";

  // Show editor section
  document.getElementById("editProductEditorSection").style.display = "block";
  document.getElementById("editProductNewPreview").style.display = "none";

  // Read file and initialize editor
  const reader = new FileReader();
  reader.onload = function (e) {
    initEditProductImageEditor(e.target.result);
  };
  reader.readAsDataURL(file);
}

// Initialize edit product image editor
function initEditProductImageEditor(imageSrc) {
  const image = document.getElementById("editProductImageToCrop");
  image.src = imageSrc;

  image.onload = function () {
    // Destroy existing cropper
    if (editProductCropper) {
      editProductCropper.destroy();
    }

    // Initialize cropper
    editProductCropper = new Cropper(image, {
      aspectRatio: 1,
      viewMode: 1,
      autoCropArea: 0.8,
      responsive: true,
      restore: true,
      guides: true,
      center: true,
      highlight: false,
      cropBoxMovable: true,
      cropBoxResizable: true,
      background: false,
      modal: false,
      toggleDragModeOnDblclick: false,
      ready: function () {
        updateEditProductPreview();
        updateEditProductSizeEstimation();
      },
      crop: function () {
        updateEditProductPreview();
      },
    });

    initEditProductAdjustmentControls();
  };
}

// Update edit product preview
function updateEditProductPreview() {
  if (!editProductCropper) return;

  const canvas = editProductCropper.getCroppedCanvas({
    width: 300,
    height: 300,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    const preview = document.getElementById("editProductCropPreview");
    const quality =
      parseInt(document.getElementById("editProductQualitySlider").value) / 100;
    preview.src = canvas.toDataURL("image/jpeg", quality);
    updateEditProductSizeEstimation();
  }
}

// Update edit product size estimation
function updateEditProductSizeEstimation() {
  if (!editProductCropper) return;

  const quality =
    parseInt(document.getElementById("editProductQualitySlider").value) / 100;
  const canvas = editProductCropper.getCroppedCanvas({
    width: 800,
    height: 800,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    const dataUrl = canvas.toDataURL("image/jpeg", quality);
    const base64Length = dataUrl.length - (dataUrl.indexOf(",") + 1);
    const padding =
      dataUrl.charAt(dataUrl.length - 2) === "="
        ? dataUrl.charAt(dataUrl.length - 1) === "="
          ? 2
          : 1
        : 0;
    const fileSize = (base64Length * 3) / 4 - padding;
    document.getElementById("editProductEstimatedSize").textContent =
      formatFileSize(fileSize);
  }
}

// Initialize edit product adjustment controls
function initEditProductAdjustmentControls() {
  // Zoom slider
  const zoomSlider = document.getElementById("editProductZoomSlider");
  const zoomValue = document.getElementById("editProductZoomValue");

  if (zoomSlider) {
    zoomSlider.addEventListener("input", function () {
      const value = parseFloat(this.value);
      zoomValue.textContent = Math.round(value * 100) + "%";
      if (editProductCropper) editProductCropper.zoomTo(value);
    });

    // Zoom buttons
    document
      .getElementById("editProductZoomIn")
      .addEventListener("click", function () {
        if (editProductCropper) {
          editProductCropper.zoom(0.1);
          const currentZoom = editProductCropper.getData().zoom;
          zoomSlider.value = currentZoom;
          zoomValue.textContent = Math.round(currentZoom * 100) + "%";
        }
      });

    document
      .getElementById("editProductZoomOut")
      .addEventListener("click", function () {
        if (editProductCropper) {
          editProductCropper.zoom(-0.1);
          const currentZoom = editProductCropper.getData().zoom;
          zoomSlider.value = currentZoom;
          zoomValue.textContent = Math.round(currentZoom * 100) + "%";
        }
      });
  }

  // Rotation controls
  const rotationSlider = document.getElementById("editProductRotationSlider");
  const rotationValue = document.getElementById("editProductRotationValue");

  if (rotationSlider) {
    rotationSlider.addEventListener("input", function () {
      const value = parseInt(this.value);
      rotationValue.textContent = value + "°";
      if (editProductCropper) editProductCropper.rotateTo(value);
    });

    document
      .getElementById("editProductRotateLeft")
      .addEventListener("click", function () {
        if (editProductCropper) {
          const current = parseInt(rotationSlider.value);
          const newValue = current - 15;
          rotationSlider.value = newValue;
          rotationValue.textContent = newValue + "°";
          editProductCropper.rotateTo(newValue);
        }
      });

    document
      .getElementById("editProductRotateRight")
      .addEventListener("click", function () {
        if (editProductCropper) {
          const current = parseInt(rotationSlider.value);
          const newValue = current + 15;
          rotationSlider.value = newValue;
          rotationValue.textContent = newValue + "°";
          editProductCropper.rotateTo(newValue);
        }
      });
  }

  // Quality slider
  const qualitySlider = document.getElementById("editProductQualitySlider");
  const qualityValue = document.getElementById("editProductQualityValue");

  if (qualitySlider) {
    qualitySlider.addEventListener("input", function () {
      const value = parseInt(this.value);
      qualityValue.textContent = value + "%";
      updateEditProductPreview();
    });
  }
}

// Apply edit product crop
function applyEditProductCrop() {
  if (!editProductCropper) return;

  const quality =
    parseInt(document.getElementById("editProductQualitySlider").value) / 100;

  // Get cropped canvas
  const canvas = editProductCropper.getCroppedCanvas({
    width: 800,
    height: 800,
    fillColor: "#fff",
    imageSmoothingEnabled: true,
    imageSmoothingQuality: "high",
  });

  if (canvas) {
    // Convert to base64
    const base64Image = canvas.toDataURL("image/jpeg", quality);

    // Set the base64 data in hidden field
    document.getElementById("editProductCroppedImageData").value = base64Image;

    // Show new image preview
    const newPreview = document.getElementById("editProductNewImage");
    newPreview.src = base64Image;

    // Switch to new preview
    document.getElementById("editProductEditorSection").style.display = "none";
    document.getElementById("editProductFileInfo").style.display = "none";
    document.getElementById("editProductNewPreview").style.display = "block";
    document.getElementById("editCurrentImagePreview").style.display = "none";
  }
}

// Initialize edit product modal events
document.addEventListener("DOMContentLoaded", function () {
  // Elements for edit modal
  const editDropArea = document.getElementById("editProductDropArea");
  const editBrowseBtn = document.getElementById("editProductBrowseBtn");
  const editImageInput = document.getElementById("editProductImageInput");
  const editChangeFileBtn = document.getElementById("editProductChangeFileBtn");
  const editCancelCropBtn = document.getElementById("editProductCancelCropBtn");
  const editApplyCropBtn = document.getElementById("editProductApplyCropBtn");
  const editResetBtn = document.getElementById("editProductResetBtn");
  const editAgainBtn = document.getElementById("editProductEditAgainBtn");
  const editChangeImageBtn = document.getElementById("editChangeImageBtn");
  const editCancelUploadBtn = document.getElementById("editCancelUploadBtn");

  // Change Image button
  if (editChangeImageBtn) {
    editChangeImageBtn.addEventListener("click", function () {
      document.getElementById("editCurrentImagePreview").style.display = "none";
      document.getElementById("editProductDropArea").style.display = "block";
    });
  }

  // Cancel upload button
  if (editCancelUploadBtn) {
    editCancelUploadBtn.addEventListener("click", function () {
      resetEditProductImageUpload();
    });
  }

  // Drag & Drop functionality for edit modal
  function preventDefaults(e) {
    e.preventDefault();
    e.stopPropagation();
  }

  function highlight() {
    editDropArea.style.borderColor = "#007bff";
    editDropArea.style.background = "#e6f7ff";
    editDropArea.style.transform = "scale(1.02)";
  }

  function unhighlight() {
    editDropArea.style.borderColor = "#ced4da";
    editDropArea.style.background = "#f8f9fa";
    editDropArea.style.transform = "scale(1)";
  }

  // Drag events for edit modal
  ["dragenter", "dragover"].forEach((eventName) => {
    editDropArea.addEventListener(
      eventName,
      function (e) {
        preventDefaults(e);
        highlight();
      },
      false
    );
  });

  ["dragleave", "drop"].forEach((eventName) => {
    editDropArea.addEventListener(
      eventName,
      function (e) {
        preventDefaults(e);
        unhighlight();
      },
      false
    );
  });

  // Handle drop for edit modal
  editDropArea.addEventListener(
    "drop",
    function (e) {
      const files = e.dataTransfer.files;
      if (files.length > 0) {
        handleEditProductFileSelect(files[0]);
      }
    },
    false
  );

  // Click events for edit modal
  if (editDropArea) {
    editDropArea.addEventListener("click", function (e) {
      if (e.target !== editBrowseBtn && e.target !== editCancelUploadBtn) {
        editImageInput.click();
      }
    });
  }

  if (editBrowseBtn) {
    editBrowseBtn.addEventListener("click", function (e) {
      e.stopPropagation();
      editImageInput.click();
    });
  }

  // File input change for edit modal
  if (editImageInput) {
    editImageInput.addEventListener("change", function (e) {
      if (this.files.length > 0) {
        handleEditProductFileSelect(this.files[0]);
      }
    });
  }

  // Change file button for edit modal
  if (editChangeFileBtn) {
    editChangeFileBtn.addEventListener("click", function () {
      resetEditProductImageUpload();
      document.getElementById("editProductDropArea").style.display = "block";
    });
  }

  // Cancel crop button for edit modal
  if (editCancelCropBtn) {
    editCancelCropBtn.addEventListener("click", function () {
      resetEditProductImageUpload();
    });
  }

  // Apply crop button for edit modal
  if (editApplyCropBtn) {
    editApplyCropBtn.addEventListener("click", applyEditProductCrop);
  }

  // Reset button for edit modal
  if (editResetBtn) {
    editResetBtn.addEventListener("click", resetEditProductAdjustments);
  }

  // Edit again button for edit modal
  if (editAgainBtn) {
    editAgainBtn.addEventListener("click", function () {
      document.getElementById("editProductNewPreview").style.display = "none";
      document.getElementById("editProductEditorSection").style.display =
        "block";
    });
  }

  // Close edit modal buttons
  document.querySelectorAll("#editProductModal .close").forEach((btn) => {
    btn.addEventListener("click", function () {
      closeEditModal();
    });
  });

  // Close modal on outside click
  window.addEventListener("click", function (event) {
    if (event.target === document.getElementById("editProductModal")) {
      closeEditModal();
    }
  });

  // Make functions globally available
  window.closeEditModal = closeEditModal;
});
