 class POSSystem {
    constructor() {
        this.cart = [];
        this.products = [];
        this.init();
    }

    async init() {
        await this.loadProducts();
        this.renderProducts();
        this.setupEventListeners();
        this.updateDateTime();
        setInterval(() => this.updateDateTime(), 1000);
    }

    async loadProducts() {
        // Simulated product data - replace with actual API call
        this.products = [
            { id: 1, name: "Keychain", price: 50, stock: 100, image: "keychain.jpg" },
            { id: 2, name: "T-Shirt", price: 299, stock: 50, image: "tshirt.jpg" },
            { id: 3, name: "Mug", price: 150, stock: 75, image: "mug.jpg" },
            { id: 4, name: "Sticker", price: 20, stock: 200, image: "sticker.jpg" },
            { id: 5, name: "Pen", price: 30, stock: 150, image: "pen.jpg" }
        ];
    }

    renderProducts() {
        const grid = document.getElementById('productsGrid');
        grid.innerHTML = this.products.map(product => `
            <div class="product-card" onclick="pos.addToCart(${product.id})">
                <img src="assets/images/${product.image}" alt="${product.name}">
                <h4>${product.name}</h4>
                <div class="price">฿${product.price}</div>
                <div class="stock">Stock: ${product.stock}</div>
            </div>
        `).join('');
    }

    addToCart(productId) {
        const product = this.products.find(p => p.id === productId);
        if (!product) return;

        const cartItem = this.cart.find(item => item.id === productId);
        
        if (cartItem) {
            if (cartItem.quantity < product.stock) {
                cartItem.quantity++;
            } else {
                alert('Not enough stock!');
                return;
            }
        } else {
            if (product.stock > 0) {
                this.cart.push({
                    ...product,
                    quantity: 1
                });
            } else {
                alert('Product out of stock!');
                return;
            }
        }

        this.updateCart();
    }

    removeFromCart(productId) {
        this.cart = this.cart.filter(item => item.id !== productId);
        this.updateCart();
    }

    updateQuantity(productId, change) {
        const cartItem = this.cart.find(item => item.id === productId);
        if (!cartItem) return;

        const product = this.products.find(p => p.id === productId);
        const newQuantity = cartItem.quantity + change;

        if (newQuantity <= 0) {
            this.removeFromCart(productId);
        } else if (newQuantity <= product.stock) {
            cartItem.quantity = newQuantity;
        } else {
            alert('Not enough stock!');
        }

        this.updateCart();
    }

    updateCart() {
        const cartItems = document.getElementById('cartItems');
        const cartTotal = document.getElementById('cartTotal');
        
        cartItems.innerHTML = this.cart.map(item => `
            <div class="cart-item">
                <span>${item.name}</span>
                <div class="quantity-controls">
                    <button class="quantity-btn" onclick="pos.updateQuantity(${item.id}, -1)">-</button>
                    <span>${item.quantity}</span>
                    <button class="quantity-btn" onclick="pos.updateQuantity(${item.id}, 1)">+</button>
                    <button class="quantity-btn btn-danger" onclick="pos.removeFromCart(${item.id})">
                        <i class="fas fa-trash"></i>
                    </button>
                </div>
                <span>฿${item.price * item.quantity}</span>
            </div>
        `).join('');

        const total = this.cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        cartTotal.textContent = `Total: ฿${total}`;
    }

    checkout() {
        if (this.cart.length === 0) {
            alert('Cart is empty!');
            return;
        }

        const billNumber = 'BILL-' + Date.now();
        const total = this.cart.reduce((sum, item) => sum + (item.price * item.quantity), 0);
        
        // Show bill preview
        this.showBill(billNumber, total);
    }

    showBill(billNumber, total) {
        const billContent = `
            <div class="bill-container">
                <div class="bill-header">
                    <h2>Souvenir Shop</h2>
                    <p>123 Shop Street, Bangkok</p>
                    <p>Tel: 02-123-4567</p>
                </div>
                <div class="bill-info">
                    <p><strong>Bill No:</strong> ${billNumber}</p>
                    <p><strong>Date:</strong> ${new Date().toLocaleString()}</p>
                </div>
                <div class="bill-items">
                    ${this.cart.map(item => `
                        <div class="bill-item">
                            <span>${item.name} x ${item.quantity}</span>
                            <span>฿${item.price * item.quantity}</span>
                        </div>
                    `).join('')}
                </div>
                <div class="bill-total">
                    <div class="bill-item">
                        <span>Total:</span>
                        <span>฿${total}</span>
                    </div>
                </div>
                <p style="text-align: center; margin-top: 1rem;">Thank you for your purchase!</p>
                <button onclick="window.print()" class="btn">Print Bill</button>
                <button onclick="pos.completeSale()" class="btn btn-success">Complete Sale</button>
            </div>
        `;

        document.body.innerHTML = billContent;
    }

    completeSale() {
        // Process sale and update stock
        alert('Sale completed successfully!');
        this.cart = [];
        window.location.reload();
    }

    updateDateTime() {
        const now = new Date();
        document.getElementById('currentDateTime').textContent = 
            now.toLocaleDateString() + ' ' + now.toLocaleTimeString();
    }

    setupEventListeners() {
        // Search functionality
        document.getElementById('productSearch')?.addEventListener('input', (e) => {
            const searchTerm = e.target.value.toLowerCase();
            const filteredProducts = this.products.filter(product => 
                product.name.toLowerCase().includes(searchTerm)
            );
            this.renderFilteredProducts(filteredProducts);
        });
    }

    renderFilteredProducts(filteredProducts) {
        const grid = document.getElementById('productsGrid');
        grid.innerHTML = filteredProducts.map(product => `
            <div class="product-card" onclick="pos.addToCart(${product.id})">
                <img src="assets/images/${product.image}" alt="${product.name}">
                <h4>${product.name}</h4>
                <div class="price">฿${product.price}</div>
                <div class="stock">Stock: ${product.stock}</div>
            </div>
        `).join('');
    }
}

// Initialize POS system
const pos = new POSSystem();
