// PWA Installation and Service Worker Registration
class PWAHelper {
  constructor() {
    this.deferredPrompt = null;
    this.init();
  }

  init() {
    // Register service worker
    if ('serviceWorker' in navigator) {
      window.addEventListener('load', () => {
        this.registerServiceWorker();
        this.setupInstallPrompt();
        this.checkNetworkStatus();
      });
    }

    // Update app when new content is available
    this.setupUpdateNotifications();
  }

  registerServiceWorker() {
    navigator.serviceWorker.register('/service-worker.js')
      .then(registration => {
        console.log('ServiceWorker registration successful');
        
        // Check for updates every hour
        setInterval(() => {
          registration.update();
        }, 60 * 60 * 1000);
      })
      .catch(error => {
        console.log('ServiceWorker registration failed: ', error);
      });
  }

  setupInstallPrompt() {
    window.addEventListener('beforeinstallprompt', (e) => {
      e.preventDefault();
      this.deferredPrompt = e;
      this.showInstallButton();
    });
  }

  showInstallButton() {
    // Create install button if it doesn't exist
    let installBtn = document.getElementById('pwa-install-btn');
    
    if (!installBtn) {
      installBtn = document.createElement('button');
      installBtn.id = 'pwa-install-btn';
      installBtn.innerHTML = '📱 Install App';
      installBtn.className = 'pwa-install-btn';
      installBtn.style.cssText = `
        position: fixed;
        bottom: 20px;
        right: 20px;
        background: #007bff;
        color: white;
        border: none;
        padding: 12px 20px;
        border-radius: 25px;
        cursor: pointer;
        box-shadow: 0 4px 12px rgba(0,123,255,0.3);
        z-index: 10000;
        font-family: Arial, sans-serif;
        font-size: 14px;
      `;
      
      document.body.appendChild(installBtn);
      
      installBtn.addEventListener('click', () => {
        this.promptInstall();
      });
    }
  }

  promptInstall() {
    if (this.deferredPrompt) {
      this.deferredPrompt.prompt();
      
      this.deferredPrompt.userChoice.then((choiceResult) => {
        if (choiceResult.outcome === 'accepted') {
          console.log('User accepted install');
        }
        this.deferredPrompt = null;
        
        // Hide install button
        const installBtn = document.getElementById('pwa-install-btn');
        if (installBtn) {
          installBtn.style.display = 'none';
        }
      });
    }
  }

  checkNetworkStatus() {
    window.addEventListener('online', () => {
      this.showNotification('Back online', 'You are now connected to the internet.');
      this.syncData();
    });

    window.addEventListener('offline', () => {
      this.showNotification('You are offline', 'Some features may be limited.');
    });

    // Initial check
    if (!navigator.onLine) {
      this.showNotification('You are offline', 'Some features may be limited.');
    }
  }

  setupUpdateNotifications() {
    let refreshing = false;
    
    navigator.serviceWorker.addEventListener('controllerchange', () => {
      if (!refreshing) {
        refreshing = true;
        this.showNotification('Update available', 'Reload to get the latest version.', true, () => {
          window.location.reload();
        });
      }
    });
  }

  syncData() {
    // Implement data synchronization when coming back online
    if ('serviceWorker' in navigator && navigator.serviceWorker.controller) {
      navigator.serviceWorker.controller.postMessage({
        type: 'SYNC_DATA'
      });
    }
  }

  showNotification(title, body, showAction = false, actionCallback = null) {
    // Use browser notifications if available
    if ('Notification' in window && Notification.permission === 'granted') {
      const notification = new Notification(title, {
        body,
        icon: '/pwa-icons/icon-192x192.png'
      });
      
      if (showAction && actionCallback) {
        notification.onclick = actionCallback;
      }
    }
    
    // Fallback to custom notification
    else {
      this.showCustomNotification(title, body, showAction, actionCallback);
    }
  }

  showCustomNotification(title, body, showAction, actionCallback) {
    const notificationEl = document.createElement('div');
    notificationEl.className = 'pwa-notification';
    notificationEl.innerHTML = `
      <strong>${title}</strong>
      <p>${body}</p>
      ${showAction ? '<button class="pwa-notification-btn">Reload</button>' : ''}
    `;
    
    notificationEl.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: white;
      border-left: 4px solid #007bff;
      padding: 15px;
      border-radius: 5px;
      box-shadow: 0 4px 12px rgba(0,0,0,0.15);
      z-index: 10001;
      max-width: 300px;
      font-family: Arial, sans-serif;
    `;
    
    document.body.appendChild(notificationEl);
    
    if (showAction) {
      const btn = notificationEl.querySelector('.pwa-notification-btn');
      btn.addEventListener('click', actionCallback);
    }
    
    setTimeout(() => {
      notificationEl.remove();
    }, 5000);
  }
}

// Initialize PWA when DOM is loaded
document.addEventListener('DOMContentLoaded', () => {
  new PWAHelper();
});